<?php
session_start();
require 'config.php';
require __DIR__ . '/vendor/autoload.php';
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

// --- التحقق من تسجيل الدخول ---
if(!isset($_SESSION['logged_in'])){
    header("Location: login.php?redirect=customers_summary.php");
    exit;
}

// --- استلام الفلاتر ---
$filter_rep     = $_GET['representative'] ?? '';
$filter_status  = $_GET['status'] ?? ''; // over_limit
$filter_overdue = $_GET['overdue'] ?? ''; // overdue only
$start_date     = $_GET['start_date'] ?? '';
$end_date       = $_GET['end_date'] ?? '';
$export         = $_GET['export'] ?? '';

// --- جلب قائمة المندوبين ---
$reps = $conn->query("SELECT DISTINCT representative FROM customers WHERE representative IS NOT NULL AND representative!=''")->fetchAll(PDO::FETCH_COLUMN);

// --- جلب العملاء ---
$sql = "SELECT id, name, balance, representative, credit_balance FROM customers WHERE 1";
$params = [];
if($filter_rep !== ''){
    $sql .= " AND representative=?";
    $params[] = $filter_rep;
}
$sql .= " ORDER BY balance DESC";
$stmt = $conn->prepare($sql);
$stmt->execute($params);
$customers_all = $stmt->fetchAll(PDO::FETCH_ASSOC);

// --- دالة حالة العميل ---
function getCustomerStatus($balance, $credit_raw){
    $credit_lower = mb_strtolower(trim($credit_raw ?? ''), 'UTF-8');
    $status = '';
    $diff = 0;
    $row_class = '';

    if ($credit_lower == 'مفتوح') {
        $status = 'حساب مفتوح';
        $row_class = 'no-limit';
    } elseif ($credit_lower == 'كاش') {
        $diff = $balance;
        $status = ($balance > 0) ? 'تجاوز الحد' : 'ضمن الحد';
        $row_class = ($balance > 0) ? 'over-limit' : 'within-limit';
    } elseif ($credit_raw === '' || !is_numeric($credit_raw)) {
        $status = 'لم يتم تحديد حد';
        $row_class = 'no-limit';
    } else {
        $credit_value = floatval($credit_raw);
        $diff = $balance - $credit_value;
        if ($balance > $credit_value) {
            $status = 'تجاوز الحد';
            $row_class = 'over-limit';
        } else {
            $status = 'ضمن الحد';
            $row_class = 'within-limit';
        }
    }
    return [$status, $diff, $row_class];
}

// --- فلترة العملاء والفواتير ---
$customers = [];
$total_balance = 0;
$total_invoices = 0;
$total_diff = 0;
$last_update = null;

foreach($customers_all as $c){
    list($status, $diff, $row_class) = getCustomerStatus($c['balance'], $c['credit_balance']);

    // فلتر الحسابات المتجاوزة
    if($filter_status==='over_limit' && $status!=='تجاوز الحد') continue;

    // جلب الفواتير
    $stmt2 = $conn->prepare("SELECT id, due_date, amount FROM invoices WHERE customer_id=? ORDER BY due_date ASC");
    $stmt2->execute([$c['id']]);
    $invoices = $stmt2->fetchAll(PDO::FETCH_ASSOC);

    // فلترة حسب التاريخ
    if($start_date && $end_date){
        $invoices = array_filter($invoices, function($inv) use($start_date,$end_date){
            $time = strtotime($inv['due_date']);
            return $time>=strtotime($start_date) && $time<=strtotime($end_date);
        });
    }

    // فلترة الفواتير المتأخرة
    if($filter_overdue==='yes'){
        $invoices = array_filter($invoices, function($inv){
            return strtotime($inv['due_date']) < time();
        });
    }

    if(empty($invoices)) continue;

    $c['status'] = $status;
    $c['diff'] = $diff;
    $c['row_class'] = $row_class;
    $c['invoices'] = $invoices;

    $customers[] = $c;

    // --- المجاميع ---
    $total_balance += $c['balance'];
    $total_diff += $diff;
    $total_invoices += array_sum(array_column($invoices,'amount'));

    // آخر تحديث
    foreach($invoices as $inv){
        $time = strtotime($inv['due_date']);
        if(!$last_update || $time > $last_update){
            $last_update = $time;
        }
    }
}

// --- تصدير Excel ---
if($export === 'yes'){
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setCellValue("A1","اسم العميل");
    $sheet->setCellValue("B1","الرصيد");
    $sheet->setCellValue("C1","المندوب");
    $sheet->setCellValue("D1","الحد الائتماني");
    $sheet->setCellValue("E1","الحالة");
    $sheet->setCellValue("F1","الفارق");
    $sheet->setCellValue("G1","الفواتير");

    $rowNum = 2;
    foreach($customers as $c){
        $invoice_text = '';
        foreach($c['invoices'] as $inv){
            $invoice_text .= $inv['due_date']." : ".number_format($inv['amount'],2)."\n";
        }
        $invoice_text = trim($invoice_text);

        $sheet->setCellValue("A$rowNum",$c['name']);
        $sheet->setCellValue("B$rowNum",$c['balance']);
        $sheet->setCellValue("C$rowNum",$c['representative']);
        $sheet->setCellValue("D$rowNum",$c['credit_balance']);
        $sheet->setCellValue("E$rowNum",$c['status']);
        $sheet->setCellValue("F$rowNum",($c['status']==='تجاوز الحد') ? $c['diff'] : '');
        $sheet->setCellValue("G$rowNum",$invoice_text);
        $rowNum++;
    }

    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="customers_summary.xlsx"');
    $writer = new Xlsx($spreadsheet);
    $writer->save('php://output');
    exit;
}

?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="UTF-8">
<title>تقرير العملاء</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
.over-limit {background-color:#f8d7da !important; font-weight:bold;}
.within-limit {background-color:#d4edda !important;}
.no-limit {background-color:#fff3cd !important;}
.table td {vertical-align: middle;}
.summary-box {background:#f8f9fa; border:1px solid #ddd; padding:12px; border-radius:8px; margin-bottom:15px;}
</style>
</head>
<body class="container mt-4">

<h2 class="mb-4 text-center">📊 تقرير العملاء </h2>

<!-- الملخص أعلى -->
<div class="summary-box">
    🧾 مجموع أرصدة العملاء: <b><?= number_format($total_balance,2) ?></b> |
    📑 مجموع الفواتير: <b><?= number_format($total_invoices,2) ?></b> |
    🔄 مجموع تجاوزات الحد الائتماني: <b><?= number_format($total_diff,2) ?></b> |
 
</div>

<form method="get" class="row mb-4 g-2">
    <div class="col-md-3">
        <select name="representative" class="form-select">
            <option value="">كل المندوبين</option>
            <?php foreach($reps as $rep): ?>
                <option value="<?= htmlspecialchars($rep) ?>" <?= $filter_rep===$rep?'selected':'' ?>><?= htmlspecialchars($rep) ?></option>
            <?php endforeach; ?>
        </select>
    </div>
    <div class="col-md-3">
        <select name="status" class="form-select">
            <option value="">كل الحسابات</option>
            <option value="over_limit" <?= $filter_status==='over_limit'?'selected':'' ?>>الحسابات المتجاوزة</option>
        </select>
    </div>
    <div class="col-md-3">
        <select name="overdue" class="form-select">
            <option value="">كل الفواتير</option>
            <option value="yes" <?= $filter_overdue==='yes'?'selected':'' ?>>الفواتير المتأخرة فقط</option>
        </select>
    </div>
    <div class="col-md-3"></div>

    <div class="col-md-3">
        <input type="date" name="start_date" value="<?= htmlspecialchars($start_date) ?>" class="form-control">
    </div>
    <div class="col-md-3">
        <input type="date" name="end_date" value="<?= htmlspecialchars($end_date) ?>" class="form-control">
    </div>
    <div class="col-md-3">
        <button type="submit" class="btn btn-primary w-100">تطبيق الفلاتر</button>
    </div>
    <div class="col-md-3">
        <a href="?<?= http_build_query($_GET) ?>&export=yes" class="btn btn-success w-100">⬇️ تصدير Excel</a>
    </div>
</form>

<div class="card">
<div class="card-body p-0">
<table class="table table-bordered table-striped text-center m-0">
<thead class="table-dark">
<tr>
<th>اسم العميل</th>
<th>الرصيد</th>
<th>المندوب</th>
<th>الحد الائتماني</th>
<th>الحالة</th>
<th>الفارق</th>
<th>الفواتير</th>
</tr>
</thead>
<tbody>
<?php if(empty($customers)): ?>
<tr><td colspan="7">⚠️ لا توجد بيانات</td></tr>
<?php else: ?>
<?php foreach($customers as $c): ?>
<tr class="<?= $c['row_class'] ?>">
    <td><?= htmlspecialchars($c['name']) ?></td>
    <td><?= number_format($c['balance'],2) ?></td>
    <td><?= htmlspecialchars($c['representative'] ?? '') ?></td>
    <td><?= htmlspecialchars($c['credit_balance'] ?? '') ?></td>
    <td><?= $c['status'] ?></td>
    <td><?= ($c['status']==='تجاوز الحد')?number_format($c['diff'],2):'-' ?></td>
    <td>
        <?php if(empty($c['invoices'])): ?>
            لا توجد فواتير
        <?php else: ?>
            <table class="table table-sm m-0">
            <?php foreach($c['invoices'] as $inv): ?>
                <tr class="<?= (strtotime($inv['due_date'])<time())?'table-danger':'' ?>">
                    <td><?= htmlspecialchars($inv['due_date']) ?></td>
                    <td><?= number_format($inv['amount'],2) ?></td>
                </tr>
            <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </td>
</tr>
<?php endforeach; ?>
<?php endif; ?>
</tbody>
</table>
</div>
</div>

<!-- الملخص أسفل -->
<div class="summary-box mt-3">
    🧾 مجموع أرصدة العملاء: <b><?= number_format($total_balance,2) ?></b> |
    📑 مجموع الفواتير: <b><?= number_format($total_invoices,2) ?></b> |
    🔄 مجموع  تجاوزات الحد الائتماني: <b><?= number_format($total_diff,2) ?></b> |
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
